
// SimParm: Simple and flexible C++ configuration framework
// Copyright (C) 2007 Australian National University
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// Contact:
// Kevin Pulo
// kevin.pulo@anu.edu.au
// Leonard Huxley Bldg 56
// Australian National University, ACT, 0200, Australia

//package simparm.twiddler;

import java.io.*;
import java.awt.*;
import java.awt.event.*;


abstract class ConfigEntry implements MouseListener {
	private String name;
	private String desc;
	private boolean invalid;
	private boolean viewable;
	private boolean editable;
	private boolean outputOnChange;
	private PrintStream output;

	protected Component label;
	protected Component field;

	public ConfigEntry() {
		this.name = "";
		this.desc = "";
		this.invalid = false;
		this.viewable = true;
		this.editable = true;
		this.outputOnChange = false;
		this.output = System.out;
	}

	public ConfigEntry(String name, String desc) {
		this.name = name;
		setDesc(desc);
		this.invalid = false;
		this.viewable = true;
		this.editable = true;
		this.outputOnChange = false;
		this.output = System.out;
	}

	public ConfigEntry(String name, String desc, boolean invalid, boolean viewable, boolean editable) {
		this.name = name;
		setDesc(desc);
		this.invalid = invalid;
		this.viewable = viewable;
		this.editable = editable;
		this.outputOnChange = false;
		this.output = System.out;
	}

	public void setName(String name) {
		this.name = name;
	}

	public void setField(Component component) {
		this.field = component;
		this.field.addMouseListener(this);
	}

	public void setLabel(Component label) {
		this.label = label;
		this.label.addMouseListener(this);
	}

	public void setDesc(String desc) {
		this.desc = desc;
		setLabel(new Label(this.desc + ":", Label.RIGHT));
	}

	public void setOutput(PrintStream output) {
		this.output = output;
	}

	public String getName() {
		return name;
	}

	public String getDesc() {
		return desc;
	}

	public boolean isEditable() {
		return editable;
	}

	public boolean isOutputOnChange() {
		return outputOnChange;
	}

	public boolean isViewable() {
		return viewable;
	}

	public abstract boolean setValueFromInputStream(String word);

	public void addToTwiddler(Twiddler twiddler) {
		if ( ! isViewable()) {
			return;
		}
		if ( ! isEditable()) {
			field.setEnabled(false);
			field.setBackground(Color.white);
		}
		if (label != null) {
			twiddler.add(label);
		} else {
			twiddler.add(new Label());
		}
		if (field != null) {
			twiddler.add(field);
		} else {
			twiddler.add(new Label());
		}
		twiddler.numControls++;
	}

	public void doPrint() {
		doPrint(output);
	}

	public void doPrint(PrintStream out) {
		//System.err.println("OUTPUTTING: " + name);
		//try {
		//	throw new Exception();
		//} catch(Exception e) {
		//	e.printStackTrace();
		//}
		out.print("set " + name + " = ");
		doSpecificPrint(out);
		out.println();
		out.flush();
	}

	public abstract void doSpecificPrint(PrintStream out);

	public static String[] readLineWords(BufferedReader input, int limit) throws IOException {
		String s = input.readLine().trim();
		if (s == null) {
			return null;
		}
		String word[] = s.split("[ 	]+", limit);
		//for (int i = 0; i < word.length; i++) {
		//	System.err.println("word[" + i + "] = " + word[i]);
		//}
		return word;
	}

	public static String[] readLineWords(BufferedReader input) throws IOException {
		String s = input.readLine();
		if (s == null) {
			return null;
		}
		String word[] = s.split("[ 	]+");
		//for (int i = 0; i < word.length; i++) {
		//	System.err.println("word[" + i + "] = " + word[i]);
		//}
		return word;
	}

	public static String readString(BufferedReader input, String expectedKeyword) throws IOException {
		String word[] = readLineWords(input, 2);
		if ( ! word[0].equals(expectedKeyword)) {
			System.err.println("Expected \"" + expectedKeyword + "\" keyword, found \"" + word[0] + "\"");
			System.exit(1);
		}
		if (word.length > 1) {
			return word[1];
		} else {
			return "";
		}
	}

	public static void readKeyword(BufferedReader input, String expectedKeyword) throws IOException {
		readString(input, expectedKeyword);
	}

	public static boolean readBool(BufferedReader input, String expectedKeyword) throws IOException {
		String s = readString(input, expectedKeyword).toLowerCase();
		return ( s.equalsIgnoreCase("1") || s.equalsIgnoreCase("true") || s.equalsIgnoreCase("t") || s.equalsIgnoreCase("yes") || s.equalsIgnoreCase("y") );
	}

	public static boolean readBoolean(BufferedReader input, String expectedKeyword) throws IOException {
		return readBool(input, expectedKeyword);
	}

	public static double readDouble(BufferedReader input, String expectedKeyword) throws IOException {
		String s = readString(input, expectedKeyword);
		try {
			return Double.parseDouble(s);
		} catch (NumberFormatException ex) {
			System.err.println("Expected double value, found \"" + s + "\"");
			System.exit(1);
			return 0.0;
		}
	}

	public static long readLong(BufferedReader input, String expectedKeyword) throws IOException {
		String s = readString(input, expectedKeyword);
		try {
			return Long.parseLong(s);
		} catch (NumberFormatException ex) {
			System.err.println("Expected long value, found \"" + s + "\"");
			System.exit(1);
			return 0;
		}
	}

	public void readDefinition(BufferedReader input) throws IOException {
		setName(readString(input, "name"));
		setDesc(readString(input, "desc"));
		invalid = readBool(input, "invalid");
		viewable = readBool(input, "viewable");
		editable = readBool(input, "editable");
		outputOnChange = readBool(input, "outputOnChange");
		readDefinitionSpecific(input);
		readKeyword(input, "end");
	}

	public abstract void readDefinitionSpecific(BufferedReader input) throws IOException;

	public void mouseClicked(MouseEvent e) {
	}

	public void mousePressed(MouseEvent e) {
	}

	public void mouseReleased(MouseEvent e) {
	}

	public void mouseEntered(MouseEvent e) {
		if (label instanceof Label) {
			label.setFont(label.getFont().deriveFont(Font.BOLD));
		}
	}

	public void mouseExited(MouseEvent e) {
		if (label instanceof Label) {
			label.setFont(label.getFont().deriveFont(Font.PLAIN));
		}
	}

}

